<?php

namespace App\Filament\Pages\Auth;

use App\Actions\Subscription\CreateSubscription;
use App\Models\Plan;
use App\Models\Subscription;
use App\Models\User;
use Carbon\Carbon;
use DanHarrin\LivewireRateLimiting\Exceptions\TooManyRequestsException;
use Filament\Events\Auth\Registered;
use Filament\Facades\Filament;
use Filament\Forms\Components\Group;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Get;
use Filament\Http\Responses\Auth\RegistrationResponse;
use Filament\Notifications\Notification;
use Filament\Pages\Auth\Register as BaseRegister;
use Illuminate\Support\Facades\Schema;
use Ysfkaya\FilamentPhoneInput\Forms\PhoneInput;

class Register extends BaseRegister
{
    /**
     * @var view-string
     */
    protected static string $view = 'filament.auth.register';

    /**
     * @return array<int | string, string | Form>
     */
    protected function getForms(): array
    {
        return [
            'form' => $this->form(
                $this->makeForm()
                    ->schema([
                        Group::make([
                            TextInput::make('first_name')
                                ->label(__('messages.staff.first_name') . ':')
                                ->validationAttribute(__('messages.staff.first_name'))
                                ->placeholder(__('messages.staff.first_name'))
                                ->required()
                                ->maxLength(255)
                                ->autofocus(),
                            TextInput::make('last_name')
                                ->label(__('messages.staff.last_name') . ':')
                                ->validationAttribute(__('messages.staff.last_name'))
                                ->placeholder(__('messages.staff.last_name'))
                                ->required()
                                ->maxLength(255),
                        ])->columns(2),
                        $this->getEmailFormComponent()->label(__('messages.staff.email') . ':')->validationAttribute(__('messages.staff.email'))->placeholder(__('messages.staff.email')),
                        TextInput::make('username')
                            ->label(__('messages.staff.username') . ':')
                            ->validationAttribute(__('messages.staff.username'))
                            ->placeholder(__('messages.staff.username'))
                            ->required()
                            ->maxLength(255),
                        $this->getPasswordFormComponent()->label(__('messages.staff.password') . ':')->validationAttribute(__('messages.staff.password'))->placeholder(__('messages.staff.password')),
                        $this->getPasswordConfirmationFormComponent()->label(__('messages.staff.confirm_password') . ':')->validationAttribute(__('messages.staff.confirm_password'))->placeholder(__('messages.staff.confirm_password')),
                    ])
                    ->statePath('data'),
            ),
        ];
    }

    protected function getFormActions(): array
    {
        return [
            $this->getRegisterFormAction()
                ->extraAttributes(['class' => 'w-full flex items-center justify-center space-x-3'])
                ->label(__('messages.common.submit')),
        ];
    }

    public function register(): ?RegistrationResponse
    {
        $this->dispatch('validationFailed');

        try {
            $this->rateLimit(5);
        } catch (TooManyRequestsException $exception) {
            $this->getRateLimitedNotification($exception)?->send();

            return null;
        }

        $user = $this->wrapInDatabaseTransaction(function () {

            $data = $this->form->getState();

            $data['contact'] = '1234567890'; // Replace with the actual contact value
            $data['gender'] = 0;
            if (Schema::hasColumn('users', 'is_default')) {
                $data['is_default'] = false;
            }
            $data['type'] = User::STAFF;

            $data = $this->mutateFormDataBeforeRegister($data);

            $user = $this->handleRegistration($data);

            $user->assignRole('customer');

            $this->form->model($user)->saveRelationships();

            return $user;
        });

        $plan = Plan::whereIsDefault(true)->first();
        Subscription::create([
            'plan_id' => $plan->id,
            'plan_amount' => $plan->price,
            'payable_amount' => $plan->price,
            'plan_frequency' => Plan::MONTHLY,
            'starts_at' => Carbon::now(),
            'ends_at' => Carbon::now()->addDays($plan->trial_days),
            'trial_ends_at' => Carbon::now()->addDays($plan->trial_days),
            'status' => Subscription::ACTIVE,
            'user_id' => $user->id,
            'no_of_post' => $plan->post_count,
        ]);

        event(new Registered($user));

        $user->sendEmailVerificationNotification();
        Notification::make()
            ->success()
            ->title(__('messages.placeholder.registered_success'))
            ->send();

        return app(RegistrationResponse::class);
    }
}
